clear

global doc "..\"
gl rawdata  "$doc\raw data"
cd "stata output"

********************************************
** Trade shares
********************************************
	
/*
	This file aims to calculate the trade shares changes with the US for 19 
	countries. Two-way trade shares refer to a country's trade with the US 
	divided by the country's total trade with the rest of the world. 
	Import/export shares refer to the the proportion of trade to the total of 
	imports/exports.
	
	
	Variables:
	us_expval: A country's exports to the US (US imports)
	us_impval: A country's imports from the US (US exports)
	total_expval: A country's exports to the world
	total_impval: A country's imports from the world
	
	Notes: Trade data with the US are reported from the US perspective. 
	All bilateral flows are converted to the partner country's perspective. 
	Therefore, US imports are treated as the partner country's exports to the US, 
	and US exports are treated as the partner country's imports from the US. 
	Variable names reflect this convention.
*/

***** 0. Starting point *****

* program to combine two text files

capture program drop import_text_file
program define import_text_file
    version 19.5
    args country file1 file2 outfile
    * file 1
    import delimited "`file1'", clear
    tempfile t1
    save `t1'
    * file 2
    import delimited "`file2'", clear
    * merge by country variable
    merge 1:1 `country' using `t1'
	keep if _m == 3
    capture drop _m
    capture drop v14
    * save
    save "`outfile'", replace
end


* program to reshape and clean data
capture program drop reshape_clean
program define reshape_clean
	version 19.5
    args prefix country outputfile
    reshape long `prefix', i(`country') j(period) string
    
	* change 2024m01 to 2024m1
    replace period = subinstr(period, "m0", "m", .)
	gen mdate = monthly(period, "YM")
    format mdate %tm
	keep if mdate >= tm(2024m1) & mdate <= tm(2025m10)
    gen year = year(dofm(mdate))
    gen month = month(dofm(mdate))
    rename `country' country
	sort mdate
	capture drop v22
    save `outputfile', replace
end


************************************************************
*                        US Data                           *
************************************************************

**# IM by the US (equivalent to the listed countries' exports from the US)

import_text_file exporters "$rawdata\imports\Trade_Map_-_List_of_supplying_markets_for_a_product_imported_by_United_States_of_America (1).txt" "$rawdata\imports\Trade_Map_-_List_of_supplying_markets_for_a_product_imported_by_United_States_of_America.txt" us_imp_temp
use us_imp_temp, clear
reshape_clean importedvaluein exporters exports_to_US


**# EX by the US (equivalent to the listed countries' imports from the US)
import_text_file importers "$rawdata\exports\Trade_Map_-_List_of_importing_markets_for_a_product_exported_by_United_States_of_America (1).txt" "$rawdata\exports\Trade_Map_-_List_of_importing_markets_for_a_product_exported_by_United_States_of_America.txt" us_exp_temp	
use us_exp_temp, clear
reshape_clean exportedvaluein importers imports_to_US

use exports_to_US, clear
merge 1:1 country period year month using imports_to_US
	rename (exportedvaluein importedvaluein) (us_impval us_expval)
	keep if _m == 3
	drop _m
	capture drop v22
	gen us_twoway = us_expval + us_impval
save US_data, replace

************************************************************
*             Total imports and total exports              *
************************************************************

**# IM - Total
import_text_file importers "$rawdata\imports\Trade_Map_-_List_of_importers_for_the_selected_product_ (1).txt" "$rawdata\imports\Trade_Map_-_List_of_importers_for_the_selected_product_.txt" total_imp_temp
use total_imp_temp, clear
reshape_clean importedvaluein importers Total_imp

**# EX - Total
import_text_file exporters "$rawdata\exports\Trade_Map_-_List_of_exporters_for_the_selected_product_ (1).txt" "$rawdata\exports\Trade_Map_-_List_of_exporters_for_the_selected_product_.txt" total_exp_temp
use total_exp_temp, clear
reshape_clean exportedvaluein exporters Total_exp

use Total_exp, clear
merge 1:1 country period year month using Total_imp
	rename (exportedvaluein importedvaluein) (total_expval total_impval)
	capture drop _m
	gen total_twoway = total_expval + total_impval
save Total_data, replace

************************************************************
*                     Calculations                         *
************************************************************
use US_data, clear
	
merge 1:1 country period year month using Total_data
	keep if _m == 3
	capture drop _m v22
	keep country us_expval total_expval us_impval total_impval us_twoway total_twoway period year month

* some countries are missing monthly data for certain months between jan–sep 2025;
* before aggregating, we want to ensure that each country has complete data for all months (jan-dec 2024 & jan-sep 2025);
* that is, we will keep only countries with no missing values in the following variables:
local var_list us_twoway total_twoway period year month
	* count missing values in each row
	egen missing_vars = rowmiss(`var_list')
	* count by country
	bys country: egen country_miss = max(missing_vars)
	* drop all observations for countries with missing values
	drop if country_miss
	drop missing_vars country_miss
	
collapse(sum) us_expval total_expval us_impval total_impval us_twoway total_twoway, by (country year)
	gen us_impshare = us_impval / total_impval
	gen us_expshare = us_expval / total_expval
	gen us_twowayshare = us_twoway / total_twoway

reshape wide us_expval total_expval us_impval total_impval us_twoway total_twoway us_impshare us_expshare us_twowayshare, i(country) j(year) 
save Final_data, replace


************************************************************
*   ---   *
************************************************************

**# Country list
insheet using "$rawdata\country list\country_list.txt", clear
	rename v1 country
	tempfile country_list
save `country_list'
use Final_data, clear
	merge m:1 country using `country_list'
	keep if _m == 3
	drop _m
save selected_country_data, replace


* export output
* 1
use US_data, clear
export excel using "output.xlsx", sheet("US", replace) firstrow(variables)

* 2
use Total_data, clear
export excel using "output.xlsx", sheet("Total", replace) firstrow(variables)

* 3
use selected_country_data, clear
export excel using "output.xlsx", sheet("Selected countries", replace) firstrow(variables) 

* 4
use selected_country_data, clear
preserve
keep country us_expval* total_expval* us_expshare*
export excel using "output.xlsx", sheet("Selected countries-EX", replace) firstrow(variables) 
restore

* 5
use selected_country_data, clear
preserve
keep country us_impval* total_impval* us_impshare*
export excel using "output.xlsx", sheet("Selected countries-IM", replace) firstrow(variables) 
restore

*6
use selected_country_data, clear
preserve
keep country us_twoway* total_twoway* us_twowayshare*
export excel using "output.xlsx", sheet("Selected countries-twoway", replace) firstrow(variables) 
restore
